/*------------------------------------------------------------------------------*
 * File Name:	ReportCreation.c		 										*
 * Creation: 	ER, 07/25/02													*
 * Purpose:		OriginC example for COM connectivity with MSOffice				*
 * Copyright (c) OriginLab Corp. 2001, 2002, 2003, 2004, 2005, 2006, 2007		*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
 
////////////////////////////////////////////////////////////////////////////////////
//
#include <origin.h>
//
////////////////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////////////////
// This example illustrates accessing MSOffice programs from Origin C using COM.
// OriginPro is required for this program to function.
// The function CreateReport() does the following:
//	1> Opens an Excel Workbook named ExcelData.XLS from the \OriginC\Samples\COM folder
//	2> Reads the data in all (2) worksheets of this sample Excel file
//	3> Graphs the data in Origin using a custom template, and performs nonlinear curve fitting
//	4> Exports the resulting graph with fit curve, to a BMP file
//	5> Creates a new Word document for each worksheets, using custom template
//	6> Places the graph image, and other relevant information into the Word document
//
// To run this function, first compile this file, then go to the Origin Script Window
// (Origin menu item: Window | Script Window), and type the following and hit Enter:
//		CreateReport
//
void CreateReport()
{
	// Bring up wait cursor
	waitCursor wCursor;	
	printf("Processing...");
	
	// All files associated with this C file are in the subfolder \Samples\COM Client\
	// under the Origin EXE path
	string strFldPath = GetAppPath(true) + "Samples\\COM Client\\";

	// Declare Excel objects
	Object	oExcel, oExcelWkbks, oExcelWkbk, oExcelWksh;
	
	// Create an Excel application object and set it as invisible
	oExcel = CreateObject("excel.application");
	oExcel.Visible = false;
	
	// Get the workbooks collection of the Excel object
	oExcelWkbks = oExcel.Workbooks;
	
	// Open the workbook with data to be analyzed
	oExcelWkbk = oExcelWkbks.Open(strFldPath+ "ExcelData.xls");
		
	// Create a Word application object and set as invisible
	Object oWord, oWordDoc;
	oWord = CreateObject("word.application");
	oWord.Visible = false;
	
	// Loop over each worksheet in the Excel workbook
	foreach (oExcelWksh in oExcelWkbk.Worksheets)
	{		
		// Read the number of points in the dataset
		int npts = oExcelWksh.Cells(7,2).Value;
		// Read start x value
		double xstart = oExcelWksh.Cells(5,2).Value;
		// Read step x value
		double xstep = oExcelWksh.Cells(6,2).Value;
	
		// Define Origin vectors to hold data
		vector vecX, vecY;
		vecX.SetSize(npts);
		vecY.SetSize(npts);

		// Fill x vector using x start and x step values
		for (int ii = 0; ii<npts; ii++)
		{
			vecX[ii] = xstart + ii * xstep;
		}

		// Read y values direclty into vector y
		vecY = oExcelWksh.Range(oExcelWksh.Cells(10,1), oExcelWksh.Cells(npts+10-1,1)).Value;	
	
		// Create an Origin worksheet
		Worksheet wksData;	
		bool bRetW = wksData.Create();

		// Declare datasets and copy data from file to datasets
		Dataset dsX(wksData,0);
		Dataset dsY(wksData,1);
		dsX = vecX;
		dsY = vecY;
		
		// Create an Origin graph, using a custom template
		GraphPage grphData;
		bool bRetG = grphData.Create(strFldPath+"Custom.OTP");
	
		// Point to active layer in current graph page
		GraphLayer grphLayer = grphData.Layers();
	
		// Declare a curve object using x,y columns of worksheet
		Curve crvData(wksData, 0, 1);
	
		// Plot data curve to active layer
		int	nPlot = grphLayer.AddPlot(crvData, IDM_PLOT_SCATTER);
		grphLayer.Rescale();

		// Get name of curve
		string strYDataName;						
		crvData.GetName(strYDataName);

		// Perform nonlinear fitting
		double centroid, width, area, offset;
		using NLSF = LabTalk.NLSF;					// Point to the NLSF object
		NLSF.Init();								// Initialize the fitter
		NLSF.Func$ = "gauss";						// Assign fitting function
		NLSF.FitData$ = strYDataName;				// Assign dataset name
		NLSF.Execute("parainit");					// Perform automatic parameter initialization
		NLSF.PasteToPlot = 0;						// Turn off pasting results to graph
		NLSF.Fit(100);								// Perform fit - up to 100 iterations
		offset = NLSF.p1;
		centroid = NLSF.p2;
		width = NLSF.p3;
		area = NLSF.p4;
		
		
		// Export graph to BMP for insertion in Word
		using Image = LabTalk.Image;
		Image.ShowOptions = 0;
		Image.FileName$ = strFldPath + "image.bmp";
		Image.Export.PageDPI( "BMP", 96, 24, 0 );
	
		// Add a new document to the Word application using custom template
		oWordDoc = oWord.Documents.Add(Template := strFldPath + "ReportTemplate.dot");

		// Insert BMP image into document
		oWordDoc.InlineShapes.AddPicture(FileName := strFldPath + "image.bmp");
	

		// Get info. on dataset, and fit parameters, and place them in table cells
		string strTemp;		
		strTemp = oExcelWksh.Cells(1,2).Value;
		oWordDoc.Tables(1).Cell(Row := 3, Column := 2).Range.InsertAfter(Text := strTemp);
		strTemp = oExcelWksh.Cells(2,2).Value;
		oWordDoc.Tables(1).Cell(Row := 5, Column := 2).Range.InsertAfter(Text := strTemp);
		string strSampleID = strTemp;
		strTemp = oExcelWksh.Cells(3,2).Value;
		oWordDoc.Tables(1).Cell(Row := 4, Column := 2).Range.InsertAfter(Text := strTemp);

		strTemp.Format( "%f", centroid);
		oWordDoc.Tables(1).Cell(Row := 7, Column := 2).Range.InsertAfter(Text := strTemp);
		strTemp.Format( "%f", width);
		oWordDoc.Tables(1).Cell(Row := 8, Column := 2).Range.InsertAfter(Text := strTemp);
		strTemp.Format( "%f", area);
		oWordDoc.Tables(1).Cell(Row := 9, Column := 2).Range.InsertAfter(Text := strTemp);
		strTemp.Format( "%f", offset);
		oWordDoc.Tables(1).Cell(Row := 10, Column := 2).Range.InsertAfter(Text := strTemp);


		// Save the Word document to a unique file with SampleID as part of file name
		oWordDoc.SaveAs(FileName := strFldPath + "Report_" + strSampleID + ".doc");
		printf("\nReport saved as %s\n",strFldPath + "Report_" + strSampleID + ".doc");

	}
	
	// Close the Excel and Word applications
	oExcel.Quit();
	oWord.Quit();
	printf("Done!\n");
}	